# main.py
import random
import string
import json
from telegram import Update, ParseMode, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import (
    Updater, CommandHandler, MessageHandler, Filters, CallbackContext, 
    CallbackQueryHandler, ConversationHandler
)

# ⚠️ خطوط ایمپورت اصلاح شده
from config import *
from database import setup_db, save_user, get_db_connection, save_product, get_product 
from keyboards import (
    get_user_main_keyboard, 
    get_admin_main_keyboard, 
    get_admin_products_keyboard,
    get_confirm_keyboard, 
    get_admin_shipping_keyboard,
    get_user_payment_keyboard, 
    get_cart_keyboard, 
    get_admin_receipt_keyboard 
)

# --- نام کاربری ربات (حتما باید تغییر کند) ---
REAL_BOT_USERNAME = "YourRealBotUsername" 

# --- استیت‌های Conversation (از config.py) ---
USER_AWAITING_RECEIPT = 100 
ADMIN_SET_SHIPPING = 200
ADMIN_RECEIPT_CONFIRM = 300
# استیت‌های افزودن محصول
PRODUCT_NAME, PRODUCT_IMAGE, PRODUCT_PRICE, PRODUCT_COLOR_OPTION, PRODUCT_INVENTORY, PRODUCT_DESC, PRODUCT_CONFIRM = range(7)
# استیت‌های ثبت سفارش
GET_FULL_NAME, GET_PHONE, GET_ADDRESS, GET_POSTAL_CODE = range(7, 11)


# --- توابع کمکی ---

def generate_product_id():
    chars = string.ascii_uppercase + string.digits
    return ''.join(random.choice(chars) for _ in range(6))

def filter_dangerous_chars(text):
    dangerous_chars = ['\u200c', '\u202e', '\u202b', '\u200d', '\u2068', '\u2069', '\xa0', '<', '>']
    clean_text = str(text) 
    for char in dangerous_chars:
        clean_text = clean_text.replace(char, ' ')
    return clean_text.strip()

# --- توابع اصلی ربات (شروع، دکمه‌ها) ---

def start(update: Update, context: CallbackContext) -> None:
    user = update.effective_user
    save_user(user.id, user.username, user.first_name)

    user_data_id = user.id 
    if user_data_id not in context.user_data: context.user_data[user_data_id] = {'cart': {}}
    if 'cart' not in context.user_data[user_data_id]: context.user_data[user_data_id]['cart'] = {}

    if context.args and context.args[0].startswith('buy_'):
        product_id = context.args[0].split('_')[1]
        context.user_data[user_data_id]['cart'][product_id] = context.user_data[user_data_id]['cart'].get(product_id, 0) + 1
        product = get_product(product_id)
        product_name = product['name'] if product else "ناشناس"

        update.message.reply_text(f"محصول **{product_name}** به سبد خرید شما اضافه شد. (لطفاً دکمه‌ها را برای ادامه استفاده کنید)",
                                  reply_markup=get_user_main_keyboard(),
                                  parse_mode=ParseMode.MARKDOWN) 
        return

    if user.id == ADMIN_ID:
        update.message.reply_text(
            f"سلام ادمین ({user.first_name})! پنل مدیریت:",
            reply_markup=get_admin_main_keyboard()
        )
    else:
        update.message.reply_text(
            f"سلام {user.first_name} عزیز! به فروشگاه خوش آمدید. برای خرید، **شناسه محصول** را ارسال کنید.",
            reply_markup=get_user_main_keyboard()
        )

def button(update: Update, context: CallbackContext) -> int:
    query = update.callback_query
    data = query.data
    user_id = query.from_user.id
    
    query.answer()
    
    try:
        if data == 'start_admin':
            query.edit_message_text(f"به پنل مدیریت بازگشتید.", reply_markup=get_admin_main_keyboard())
            return ConversationHandler.END

        if data == 'start_user':
            query.edit_message_text(f"به منوی اصلی بازگشتید.", reply_markup=get_user_main_keyboard())
            return ConversationHandler.END

        if data == 'ignore' or data == 'ignore_name' or data == 'ignore_card_number':
            query.edit_message_text("❌ عملیات لغو شد.", reply_markup=get_user_main_keyboard())
            return ConversationHandler.END

        if user_id == ADMIN_ID:
            if data == 'admin_products':
                query.edit_message_text("مدیریت محصولات:", reply_markup=get_admin_products_keyboard())
                return ConversationHandler.END
            
        else: 
            if data == 'social_media': 
                query.edit_message_text(SOCIAL_MEDIA_TEXT, reply_markup=get_user_main_keyboard(), parse_mode=ParseMode.HTML)
                return ConversationHandler.END

            elif data == 'faq':
                query.edit_message_text(FAQ_TEXT, reply_markup=get_user_main_keyboard())
                return ConversationHandler.END
                
            elif data.startswith('cart_add_'):
                product_id = data.split('_')[2]
                return update_cart(query, context, product_id, 1) 
            
            elif data.startswith('cart_remove_'):
                product_id = data.split('_')[2]
                return update_cart(query, context, product_id, -1) 
            
        return ConversationHandler.END
    
    except Exception as e:
        error_message = f"❌ خطای ناگهانی در هندلر دکمه Inline: {e} | داده دکمه: {data} | کاربر: {user_id}"
        context.bot.send_message(chat_id=ADMIN_ID, text=error_message)
        try: query.edit_message_text("❌ خطایی در اجرای درخواست شما رخ داد. ادمین مطلع شد.")
        except Exception: context.bot.send_message(chat_id=user_id, text="❌ خطایی در اجرای درخواست شما رخ داد. ادمین مطلع شد.")
        return ConversationHandler.END

# --- توابع مدیریت سبد خرید (User) ---

def display_cart_callback(update: Update, context: CallbackContext) -> int:
    query = update.callback_query
    query.answer()
    return display_cart(update, context) 

def update_cart(query: Update, context: CallbackContext, product_id: str, change: int):
    user_id = query.from_user.id
    if user_id not in context.user_data: context.user_data[user_id] = {'cart': {}}
    cart = context.user_data[user_id].get('cart', {})
    current_quantity = cart.get(product_id, 0)
    new_quantity = current_quantity + change
    
    if new_quantity > 0:
        cart[product_id] = new_quantity
    elif new_quantity <= 0 and current_quantity > 0:
        cart.pop(product_id, None)

    context.user_data[user_id]['cart'] = cart 
    return display_cart(query, context)


def display_cart(update: Update, context: CallbackContext) -> int:
    if update.callback_query:
        query = update.callback_query
        user_id = query.from_user.id
        query.answer()
        edit_func = query.edit_message_text
    else:
        user_id = update.message.from_user.id
        edit_func = update.message.reply_text 
        
    cart = context.user_data.get(user_id, {}).get('cart', {})
    
    if not cart:
        text = "سبد خرید شما خالی است. لطفاً **شناسه محصول** را برای اضافه کردن کپی و ارسال کنید."
        if update.callback_query:
             edit_func(text, reply_markup=get_user_main_keyboard(), parse_mode=ParseMode.MARKDOWN)
        else:
             edit_func(text, reply_markup=get_user_main_keyboard(), parse_mode=ParseMode.MARKDOWN)
        return ConversationHandler.END 
        
    cart_summary = "🛒 **سبد خرید شما:**\n\n"
    total_price = 0
    product_count = 0
    
    for product_id, quantity in cart.items():
        product = get_product(product_id) 
        if product:
            product_name = product['name']
            cart_summary += f" - {product_name}: **{quantity} عدد**\n"
            total_price += product['price'] * quantity
            product_count += quantity
    
    context.user_data[user_id]['order_data'] = {'total_price': total_price, 'product_count': product_count, 'cart': cart}
    
    cart_summary += f"\n**جمع کل محصولات: {total_price:,} تومان**"
    
    if update.callback_query:
        edit_func(
            cart_summary,
            reply_markup=get_cart_keyboard(cart),
            parse_mode=ParseMode.MARKDOWN
        )
    else: 
         update.message.reply_text(
            cart_summary,
            reply_markup=get_cart_keyboard(cart),
            parse_mode=ParseMode.MARKDOWN
        )
        
    return ConversationHandler.END 


# --- توابع مربوط به اطلاعات بانکی (اصلاح شده) ---

def send_bank_info(update: Update, context: CallbackContext) -> int:
    """هندلر Callback برای نمایش اطلاعات کارت بانکی و دکمه ارسال رسید"""
    query = update.callback_query
    query.answer()
    
    text = (
        "💳 **اطلاعات کارت جهت واریز وجه:**\n\n"
        f"شماره کارت: `{BANK_CARD_NUMBER}`\n"
        f"به نام: **{BANK_CARD_NAME}**\n\n"
        "لطفاً مبلغ نهایی را واریز نموده و دکمه **'ارسال رسید پرداخت'** را بزنید."
    )
    
    # 💥 اصلاح حیاتی: نمایش دکمه‌های پرداخت (شامل دکمه ارسال رسید)
    query.edit_message_text(
        text, 
        parse_mode=ParseMode.MARKDOWN, 
        reply_markup=get_user_payment_keyboard() 
    )
    return ConversationHandler.END


# --- هندلر دریافت شناسه محصول از کاربر (اصلاح شده) ---

def handle_product_id_input(update: Update, context: CallbackContext) -> None:
    product_id = update.message.text.strip()
    user_id = update.effective_user.id
    
    product = get_product(product_id)
    if not product:
        update.message.reply_text("❌ محصولی با این شناسه پیدا نشد. لطفاً شناسه را مجدداً بررسی کنید.")
        return

    if user_id not in context.user_data: context.user_data[user_id] = {'cart': {}}
    context.user_data[user_id]['cart'][product_id] = context.user_data[user_id]['cart'].get(product_id, 0) + 1
    
    product_name = product['name']

    update.message.reply_text(f"✅ محصول **{product_name}** به سبد خرید شما اضافه شد. اکنون می‌توانید شناسه محصول بعدی را ارسال کنید یا **سبد خرید من** را بزنید.",
                              reply_markup=get_user_main_keyboard(),
                              parse_mode=ParseMode.MARKDOWN) 


# --- توابع فرآیند افزودن محصول (Admin) (کوتاه شده) --- 
def start_add_product(update: Update, context: CallbackContext) -> int:
    query = update.callback_query
    query.answer()
    query.edit_message_text("لطفاً **نام محصول** را وارد کنید:")
    context.user_data['new_product'] = {'product_id': generate_product_id(), 'random_color': False}
    return PRODUCT_NAME 

def get_product_name(update: Update, context: CallbackContext) -> int:
    context.user_data['new_product']['name'] = update.message.text
    update.message.reply_text("نام محصول ثبت شد. لطفاً **تصویر محصول** را ارسال کنید:")
    return PRODUCT_IMAGE

def get_product_image(update: Update, context: CallbackContext) -> int:
    file_id = None
    if update.message.photo: file_id = update.message.photo[-1].file_id
    elif update.message.document and update.message.document.mime_type.startswith('image/'): file_id = update.message.document.file_id
    if file_id:
        context.user_data['new_product']['images'] = file_id
        update.message.reply_text("تصویر ثبت شد. لطفاً **قیمت محصول (به تومان)** را به عدد وارد کنید:")
        return PRODUCT_PRICE

def get_product_price(update: Update, context: CallbackContext) -> int:
    try:
        price = int(update.message.text)
        context.user_data['new_product']['price'] = price
        keyboard = [[InlineKeyboardButton("بله", callback_data='color_yes')], [InlineKeyboardButton("خیر", callback_data='color_no')]]
        update.message.reply_text("آیا رنگ‌بندی این محصول رندوم ارسال می‌شود؟", reply_markup=InlineKeyboardMarkup(keyboard))
        return PRODUCT_COLOR_OPTION
    except ValueError:
        update.message.reply_text("لطفاً قیمت را فقط به صورت عدد وارد کنید.")
        return PRODUCT_PRICE

def get_product_color_option(update: Update, context: CallbackContext) -> int:
    query = update.callback_query
    query.answer()
    if query.data == 'color_yes': context.user_data['new_product']['random_color'] = True
    query.edit_message_text("وضعیت رنگ‌بندی ثبت شد. لطفاً **تعداد موجودی** محصول را به عدد وارد کنید:")
    return PRODUCT_INVENTORY

def get_product_inventory(update: Update, context: CallbackContext) -> int:
    try:
        inventory = int(update.message.text)
        context.user_data['new_product']['inventory'] = inventory
        update.message.reply_text("موجودی ثبت شد. لطفاً **توضیحات کامل محصول** را وارد کنید:")
        return PRODUCT_DESC
    except ValueError:
        update.message.reply_text("لطفاً موجودی را فقط به صورت عدد وارد کنید.")
        return PRODUCT_INVENTORY

def get_product_desc(update: Update, context: CallbackContext) -> int:
    context.user_data['new_product']['description'] = update.message.text
    product_data = context.user_data['new_product']
    summary = (
        "**خلاصه‌ی محصول برای انتشار:**\n"
        f"کد شناسه: `{product_data['product_id']}`\n"
        f"نام: {product_data['name']}\n"
        f"قیمت: {product_data['price']:,} تومان\n"
        f"موجودی: {product_data['inventory']} عدد\n"
    )
    update.message.reply_text(summary, reply_markup=get_confirm_keyboard('prod_confirm'), parse_mode=ParseMode.MARKDOWN)
    return PRODUCT_CONFIRM

def confirm_product_publish(update: Update, context: CallbackContext) -> int:
    query = update.callback_query
    query.answer()
    product_data = context.user_data['new_product']
    if query.data == 'prod_confirm_yes':
        try:
            save_product(product_data)
            filtered_description = filter_dangerous_chars(product_data['description'])
            caption = (
                f"**{product_data['name']}**\n\n"
                f"💰 قیمت: {product_data['price']:,} تومان\n"
                f"📋 توضیحات:\n`{filtered_description}`\n"
                f"🎨 رنگ‌بندی: **{'❌ رنگ‌بندی رندوم ارسال می‌شود ❌' if product_data['random_color'] else '✅ رنگ‌بندی ثابت'}**\n\n"
                f"**برای سفارش، شناسه زیر را کپی و به ربات @{REAL_BOT_USERNAME} ارسال کنید.**\n"
                f"شناسه: `{product_data['product_id']}`" 
            )
            empty_keyboard = InlineKeyboardMarkup([]) 
            context.bot.send_photo(
                chat_id=CHANNEL_USERNAME,
                photo=product_data['images'],
                caption=caption,
                parse_mode=ParseMode.MARKDOWN, 
                reply_markup=empty_keyboard 
            )
            query.edit_message_text("✅ محصول با موفقیت ذخیره و در کانال منتشر شد.")
        except Exception as e:
            error_message = f"❌ خطای انتشار در کانال: {e}"
            context.bot.send_message(chat_id=ADMIN_ID, text=error_message)
            query.edit_message_text(f"❌ محصول ذخیره شد اما **انتشار در کانال ناموفق** بود.")
    else:
        query.edit_message_text("❌ افزودن محصول لغو شد.")
    context.user_data.pop('new_product', None)
    return ConversationHandler.END


# --- توابع فرآیند ثبت سفارش (User) (کوتاه شده) --- 
def confirm_checkout_start(update: Update, context: CallbackContext) -> int:
    query = update.callback_query
    query.answer()
    user_id = query.from_user.id
    if not context.user_data.get(user_id, {}).get('cart', {}):
        query.edit_message_text("سبد خرید شما خالی است و نمی‌توانید نهایی کنید.", reply_markup=get_user_main_keyboard())
        return ConversationHandler.END

    if user_id not in context.user_data: context.user_data[user_id] = {}
    query.edit_message_text("لطفاً **نام و نام خانوادگی** خود را وارد کنید:")
    return GET_FULL_NAME 

def get_full_name(update: Update, context: CallbackContext) -> int:
    user_id = update.effective_user.id
    full_name = filter_dangerous_chars(update.message.text)
    context.user_data[user_id]['order_data']['full_name'] = full_name
    update.message.reply_text("متشکرم. لطفاً **شماره تماس** خود را وارد کنید:")
    return GET_PHONE

def get_phone(update: Update, context: CallbackContext) -> int:
    user_id = update.effective_user.id
    phone = filter_dangerous_chars(update.message.text)
    context.user_data[user_id]['order_data']['phone'] = phone
    update.message.reply_text("لطفاً **آدرس دقیق** پستی خود را وارد کنید:")
    return GET_ADDRESS

def get_address(update: Update, context: CallbackContext) -> int:
    user_id = update.effective_user.id
    address = filter_dangerous_chars(update.message.text)
    context.user_data[user_id]['order_data']['address'] = address
    update.message.reply_text("لطفاً **کد پستی** خود را وارد کنید:")
    return GET_POSTAL_CODE

def get_postal_code(update: Update, context: CallbackContext) -> int:
    user = update.effective_user 
    user_id = user.id
    order_data = context.user_data[user_id]['order_data']
    postal_code_input = update.message.text.strip() 
    try:
        order_data['postal_code'] = postal_code_input
        order_data['shipping_cost'] = 0 
        order_id = random.randint(1000, 9999) 
        order_data['order_id'] = order_id
        
        context.bot_data[f'await_shipping_{order_id}'] = {
            'user_id': user_id, 
            'order_data': order_data, 
            'order_id': order_id,
            'status': 'AWAITING_SHIPPING_FEE'
        } 
        
        admin_alert_text = (
            f"🔔 سفارش جدید آماده تأیید هزینه ارسال است! (سفارش #{order_id})\n\n"
            f"مشتری: {order_data['full_name']} (ID: {user_id})\n"
            f"آدرس: {order_data['address']}\n"
            f"کد پستی: {order_data['postal_code']}\n"
            f"شماره تماس: {order_data['phone']}\n"
            f"جمع محصولات: {order_data['total_price']:,} تومان"
        )
        
        context.bot.send_message(
            chat_id=ADMIN_ID,
            text=admin_alert_text,
            reply_markup=get_admin_shipping_keyboard(order_id, action='set_shipping'), 
            parse_mode=None
        )
        
        update.message.reply_text(
            f"✅ اطلاعات شما ثبت شد. **لطفاً منتظر تأیید ادمین و تعیین هزینه ارسال باشید.**"
        )
        
        context.user_data.pop(user_id, None) 
        return ConversationHandler.END 

    except Exception as e:
        error_message = f"❌ خطای ناگهانی در ثبت کد پستی: {e} | ورودی کاربر: {postal_code_input}"
        context.bot.send_message(chat_id=ADMIN_ID, text=error_message)
        update.message.reply_text("❌ خطایی در ثبت کد پستی رخ داد. ادمین مطلع شد. لطفاً مجدداً فرآیند سفارش را شروع کنید.")
        return ConversationHandler.END 


# --- توابع مدیریت سفارش توسط ادمین (Admin Shipping) ---

def admin_shipping_handler(update: Update, context: CallbackContext) -> int:
    query = update.callback_query
    query.answer()
    data = query.data
    
    parts = data.split('_')
    action = parts[1]
    order_id = int(parts[2]) 
    
    temp_key = f'await_shipping_{order_id}'
    order_data_temp = context.bot_data.get(temp_key) 

    if not order_data_temp:
        query.edit_message_text("❌ اطلاعات این سفارش پیدا نشد.")
        return ConversationHandler.END

    if action == 'set':
        context.user_data[ADMIN_ID] = {'awaiting_shipping': {'order_id': order_id, 'user_id': order_data_temp['user_id']}}
        query.edit_message_text(f"لطفاً **هزینه ارسال** برای سفارش #{order_id} (به تومان) را به صورت **عدد** وارد کنید:")
        return ADMIN_SET_SHIPPING
    
    elif action == 'cancel_order':
        user_id_to_send = order_data_temp['user_id']
        context.bot.send_message(chat_id=user_id_to_send, text=f"❌ سفارش شما (ID: {order_id}) توسط ادمین لغو شد.", reply_markup=get_user_main_keyboard())
        query.edit_message_text(f"❌ سفارش #{order_id} لغو شد.", reply_markup=get_admin_main_keyboard())
        context.bot_data.pop(temp_key, None)
        return ConversationHandler.END
    
    return ConversationHandler.END

def admin_get_shipping_fee(update: Update, context: CallbackContext) -> int:
    user_id = update.effective_user.id
    shipping_fee_input = update.message.text.strip()
    
    if 'awaiting_shipping' not in context.user_data.get(user_id, {}):
        update.message.reply_text("❌ خطایی رخ داد. لطفاً از پنل مدیریت مجدداً اقدام کنید.", reply_markup=get_admin_main_keyboard())
        return ConversationHandler.END

    order_id = context.user_data[user_id]['awaiting_shipping']['order_id']
    temp_key = f'await_shipping_{order_id}'
    order_data_temp = context.bot_data.get(temp_key) 
    
    if not order_data_temp:
        update.message.reply_text("❌ اطلاعات این سفارش پیدا نشد. لطفاً مجدداً اقدام کنید.", reply_markup=get_admin_main_keyboard())
        context.user_data[user_id].pop('awaiting_shipping', None)
        return ConversationHandler.END

    if not shipping_fee_input.isdigit():
        update.message.reply_text("❌ لطفاً هزینه ارسال را فقط به صورت **عدد** وارد کنید:")
        return ADMIN_SET_SHIPPING
        
    try:
        shipping_fee = int(shipping_fee_input)
        order_data_temp['order_data']['shipping_cost'] = shipping_fee
        send_payment_info_to_user(context, order_id, order_data_temp)
        
        update.message.reply_text(f"✅ هزینه ارسال **{shipping_fee:,} تومان** برای سفارش #{order_id} تعیین و اطلاعات پرداخت برای مشتری ارسال شد.", 
                                  reply_markup=get_admin_main_keyboard())
                                  
        context.user_data[user_id].pop('awaiting_shipping', None)
        return ConversationHandler.END
        
    except Exception as e:
        update.message.reply_text(f"❌ خطای داخلی: {e}. لطفاً مجدداً اقدام کنید.")
        return ADMIN_SET_SHIPPING

def send_payment_info_to_user(context: CallbackContext, order_id: int, order_data_temp: dict):
    
    total_products_price = order_data_temp['order_data']['total_price']
    shipping_cost = order_data_temp['order_data']['shipping_cost'] 
    user_id_to_send = order_data_temp['user_id']
    final_price = total_products_price + shipping_cost

    payment_message = (
        f"🔔 **تأیید نهایی سفارش شما** (سفارش #{order_id})\n\n"
        f"مبلغ محصولات: {total_products_price:,} تومان\n"
        f"هزینه ارسال: **{shipping_cost:,} تومان**\n"
        f"**مبلغ نهایی قابل پرداخت: {final_price:,} تومان**\n\n"
        f"لطفاً مبلغ فوق را به شماره کارت واریز نموده و رسید پرداخت را ارسال کنید."
    )
    
    context.bot.send_message(
        chat_id=user_id_to_send, 
        text=payment_message,
        reply_markup=get_user_payment_keyboard(), 
        parse_mode=ParseMode.MARKDOWN
    )
    
    context.bot_data[f'receipt_awaiting_{user_id_to_send}'] = {
        'order_id': order_id,
        'order_data': order_data_temp['order_data'], 
        'final_price': final_price
    }
    
    temp_key = f'await_shipping_{order_id}'
    context.bot_data.pop(temp_key, None)


# --- توابع دریافت رسید از مشتری (User Receipt) (اصلاح شده) ---

def start_receipt_upload(update: Update, context: CallbackContext) -> int:
    query = update.callback_query
    query.answer()
    user_id = query.from_user.id
    
    if not context.bot_data.get(f'receipt_awaiting_{user_id}'):
        query.edit_message_text("❌ در حال حاضر سفارشی برای ارسال رسید ندارید.", reply_markup=get_user_main_keyboard())
        return ConversationHandler.END
        
    # 💥 اصلاح: حذف دکمه‌ها هنگام درخواست عکس
    query.edit_message_text("لطفاً **عکس رسید پرداخت** را ارسال کنید.", reply_markup=InlineKeyboardMarkup([]))
    
    return USER_AWAITING_RECEIPT 

def get_receipt_photo(update: Update, context: CallbackContext) -> int:
    user = update.effective_user
    user_id = user.id
    receipt_data = context.bot_data.get(f'receipt_awaiting_{user_id}')
    
    if not receipt_data:
        update.message.reply_text("❌ خطایی رخ داده است. لطفاً مجدداً از دکمه پرداخت اقدام کنید.", reply_markup=get_user_main_keyboard())
        return ConversationHandler.END

    if update.message.photo:
        receipt_file_id = update.message.photo[-1].file_id
        current_order_id = receipt_data['order_id']
        
        caption = (
            f"🔔 **رسید پرداخت جدید دریافت شد!** (سفارش #{current_order_id})\n\n"
            f"مشتری: {user.first_name} (ID: {user_id})\n"
            f"مبلغ واریز شده (نهایی): **{receipt_data['final_price']:,} تومان**\n"
        )
        
        context.bot_data[f'receipt_confirm_{current_order_id}'] = {
            'receipt_file_id': receipt_file_id,
            'user_id': user_id,
            'order_data': receipt_data['order_data'],
            'final_price': receipt_data['final_price'],
            'full_name': user.first_name
        }
        
        context.bot.send_photo(
            chat_id=ADMIN_ID,
            photo=receipt_file_id,
            caption=caption,
            reply_markup=get_admin_receipt_keyboard(current_order_id),
            parse_mode=ParseMode.MARKDOWN
        )
        
        update.message.reply_text("✅ رسید شما با موفقیت برای ادمین ارسال شد. **لطفاً منتظر تأیید نهایی ادمین باشید.**", reply_markup=get_user_main_keyboard())
        context.bot_data.pop(f'receipt_awaiting_{user_id}', None)
        return ConversationHandler.END
    
    elif update.message.text:
        if update.message.text.lower() in ['/cancel', 'لغو']:
             update.message.reply_text("❌ ارسال رسید لغو شد.", reply_markup=get_user_main_keyboard())
             context.bot_data.pop(f'receipt_awaiting_{user_id}', None)
             return ConversationHandler.END
        else:
             update.message.reply_text("لطفاً فقط **عکس رسید پرداخت** را ارسال کنید یا برای لغو، /cancel را بزنید.")
             return USER_AWAITING_RECEIPT 

    else:
        update.message.reply_text("لطفاً فقط **عکس رسید پرداخت** را ارسال کنید.")
        return USER_AWAITING_RECEIPT

# --- توابع تایید نهایی رسید (Admin Confirmation) ---

def format_order_for_channel(confirm_data: dict, context: CallbackContext) -> str:
    order_data = confirm_data['order_data']
    order_id = order_data['order_id']
    
    summary = f"📦 **سفارش جدید: #{order_id} (تایید شده)**\n\n"
    summary += "--- **اطلاعات مشتری** ---\n"
    summary += f"👤 نام و نام خانوادگی: {order_data['full_name']}\n"
    summary += f"📞 شماره تماس: {order_data['phone']}\n"
    summary += f"📬 آدرس کامل: {order_data['address']}\n"
    summary += f"📮 کد پستی: {order_data['postal_code']}\n\n"
    summary += "--- **جزئیات خرید** ---\n"
    
    for product_id, quantity in order_data['cart'].items():
        product = get_product(product_id) 
        product_name = product['name'] if product else f"محصول ناشناس ({product_id})"
        summary += f" - {product_name}: **{quantity} عدد**\n"

    summary += "\n--- **مالی** ---\n"
    summary += f"💰 جمع محصولات: {order_data['total_price']:,} تومان\n"
    summary += f"🚛 هزینه ارسال: {order_data['shipping_cost']:,} تومان\n"
    summary += f"💵 **مبلغ نهایی پرداخت شده: {confirm_data['final_price']:,} تومان**"
    
    return summary

def admin_receipt_confirm_handler(update: Update, context: CallbackContext) -> int:
    query = update.callback_query
    query.answer()
    data = query.data
    
    parts = data.split('_')
    action = parts[2]
    order_id = int(parts[3]) 
    
    temp_key = f'receipt_confirm_{order_id}'
    confirm_data = context.bot_data.get(temp_key) 
    
    if not confirm_data:
        query.edit_message_text("❌ اطلاعات این رسید پیدا نشد.")
        return ConversationHandler.END

    user_id_to_send = confirm_data['user_id']
    
    if action == 'accept':
        order_summary_text = format_order_for_channel(confirm_data, context)
        context.bot.send_message(
            chat_id=ORDER_CHANNEL_ID,
            text=order_summary_text,
            parse_mode=ParseMode.MARKDOWN
        )

        context.bot.send_message(
            chat_id=user_id_to_send, 
            text=f"✅ **رسید پرداخت شما برای سفارش #{order_id} توسط ادمین تأیید شد.** سفارش شما وارد فرآیند ارسال شد. با تشکر از خرید شما!",
            reply_markup=get_user_main_keyboard()
        )
        query.edit_message_text(f"✅ رسید سفارش #{order_id} تأیید شد و پیام نهایی برای مشتری و کانال سفارشات ارسال گردید.", reply_markup=get_admin_main_keyboard())
        
    elif action == 'reject':
        context.bot.send_message(
            chat_id=user_id_to_send, 
            text=f"❌ **رسید پرداخت شما برای سفارش #{order_id} توسط ادمین رد شد.** لطفاً رسید صحیح را مجدداً ارسال کنید. (با زدن دکمه پرداخت می‌توانید فرآیند ارسال رسید را تکرار کنید)",
            reply_markup=get_user_main_keyboard()
        )
        query.edit_message_text(f"❌ رسید سفارش #{order_id} رد شد و به مشتری اطلاع داده شد.", reply_markup=get_admin_main_keyboard())

    context.bot_data.pop(temp_key, None)
    return ConversationHandler.END


# 💡 تابع جدید برای حذف دستی دکمه (Admin) 
def remove_inline_buttons(update: Update, context: CallbackContext):
    if update.effective_user.id != ADMIN_ID:
        update.message.reply_text("❌ فقط ادمین می‌تواند از این دستور استفاده کند.")
        return
        
    try:
        message_id = update.message.reply_to_message.message_id
        
        context.bot.edit_message_reply_markup(
            chat_id=CHANNEL_USERNAME, 
            message_id=message_id, 
            reply_markup=InlineKeyboardMarkup([]) 
        )
        
        update.message.reply_text("✅ دکمه‌های Inline در پیام فوق با موفقیت حذف شدند.")

    except Exception as e:
        update.message.reply_text(f"❌ خطا در حذف دکمه‌ها. مطمئن شوید در پاسخ به پیام کانال دستور را ارسال می‌کنید. خطا: {e}")

# --- تابع اصلی برای اجرای ربات ---
def main() -> None:
    setup_db()
    updater = Updater(BOT_TOKEN, use_context=True)
    dp = updater.dispatcher

    product_conv_handler = ConversationHandler(
        entry_points=[CallbackQueryHandler(start_add_product, pattern='^admin_add_product_start$')], 
        states={
            PRODUCT_NAME: [MessageHandler(Filters.text & ~Filters.command, get_product_name)], 
            PRODUCT_IMAGE: [MessageHandler(Filters.photo | Filters.document.image, get_product_image)], 
            PRODUCT_PRICE: [MessageHandler(Filters.text & ~Filters.command, get_product_price)],
            PRODUCT_COLOR_OPTION: [CallbackQueryHandler(get_product_color_option, pattern='^color_')],
            PRODUCT_INVENTORY: [MessageHandler(Filters.text & ~Filters.command, get_product_inventory)],
            PRODUCT_DESC: [MessageHandler(Filters.text & ~Filters.command, get_product_desc)],
            PRODUCT_CONFIRM: [CallbackQueryHandler(confirm_product_publish, pattern='^prod_confirm_')]
        },
        fallbacks=[CommandHandler('cancel', start)]
    )
    
    admin_set_shipping_conv_handler = ConversationHandler(
        entry_points=[
            CallbackQueryHandler(admin_shipping_handler, pattern='^shipping_(set|cancel_order)_(\d+)$')
        ],
        states={
            ADMIN_SET_SHIPPING: [MessageHandler(Filters.text & ~Filters.command, admin_get_shipping_fee)]
        },
        fallbacks=[CommandHandler('cancel', start)]
    )

    checkout_conv_handler = ConversationHandler(
        entry_points=[
            CallbackQueryHandler(confirm_checkout_start, pattern='^confirm_checkout$')
        ],
        states={
            GET_FULL_NAME: [MessageHandler(Filters.text & ~Filters.command, get_full_name)],
            GET_PHONE: [MessageHandler(Filters.text & ~Filters.command, get_phone)],
            GET_ADDRESS: [MessageHandler(Filters.text & ~Filters.command, get_address)],
            GET_POSTAL_CODE: [MessageHandler(Filters.text & ~Filters.command, get_postal_code)],
        },
        fallbacks=[CommandHandler('cancel', start)]
    )
    
    receipt_conv_handler = ConversationHandler(
        entry_points=[
            CallbackQueryHandler(start_receipt_upload, pattern='^send_receipt_start$')
        ],
        states={
            USER_AWAITING_RECEIPT: [MessageHandler(Filters.photo | Filters.text, get_receipt_photo)]
        },
        fallbacks=[CommandHandler('cancel', start)]
    )
    
    receipt_confirm_handler = CallbackQueryHandler(admin_receipt_confirm_handler, pattern='^receipt_confirm_(accept|reject)_(\d+)$')


    # --- افزودن هندلرها (ترتیب حیاتی) ---
    dp.add_handler(product_conv_handler) 
    dp.add_handler(admin_set_shipping_conv_handler) 
    dp.add_handler(checkout_conv_handler)
    dp.add_handler(receipt_conv_handler)
    dp.add_handler(receipt_confirm_handler) 
    
    dp.add_handler(CallbackQueryHandler(display_cart_callback, pattern='^show_cart$')) 
    dp.add_handler(CallbackQueryHandler(send_bank_info, pattern='^bank_info_display$'))
    
    dp.add_handler(MessageHandler(Filters.regex(r'^[A-Z0-9]{6}$'), handle_product_id_input))
    
    dp.add_handler(CommandHandler("start", start))
    dp.add_handler(CommandHandler("admin", start))
    dp.add_handler(CommandHandler("removebuttons", remove_inline_buttons))
    
    dp.add_handler(CallbackQueryHandler(button))

    print("ربات در حال اجرا...")
    updater.start_polling()
    updater.idle()

if __name__ == '__main__':
    main()